import type { SessionStartedMessage, PartialTranscriptMessage, CommittedTranscriptMessage, CommittedTranscriptWithTimestampsMessage, ScribeErrorMessage, ScribeAuthErrorMessage } from "@elevenlabs/types";
export type { SessionStartedMessage, PartialTranscriptMessage, CommittedTranscriptMessage, CommittedTranscriptWithTimestampsMessage, ScribeErrorMessage, ScribeAuthErrorMessage, };
export type WebSocketMessage = SessionStartedMessage | PartialTranscriptMessage | CommittedTranscriptMessage | CommittedTranscriptWithTimestampsMessage | ScribeErrorMessage | ScribeAuthErrorMessage;
/**
 * Events emitted by the RealtimeConnection.
 */
export declare enum RealtimeEvents {
    /** Emitted when the session is successfully started */
    SESSION_STARTED = "session_started",
    /** Emitted when a partial (interim) transcript is available */
    PARTIAL_TRANSCRIPT = "partial_transcript",
    /** Emitted when a final transcript is available */
    COMMITTED_TRANSCRIPT = "committed_transcript",
    /** Emitted when a final transcript with timestamps is available */
    COMMITTED_TRANSCRIPT_WITH_TIMESTAMPS = "committed_transcript_with_timestamps",
    /** Emitted when an authentication error occurs */
    AUTH_ERROR = "auth_error",
    /** Emitted when an error occurs */
    ERROR = "error",
    /** Emitted when the WebSocket connection is opened */
    OPEN = "open",
    /** Emitted when the WebSocket connection is closed */
    CLOSE = "close"
}
/**
 * Manages a real-time transcription WebSocket connection.
 *
 * @example
 * ```typescript
 * const connection = await Scribe.connect({
 *     token: "...",
 *     modelId: "scribe_v2_realtime",
 *     audioFormat: AudioFormat.PCM_16000,
 *     sampleRate: 16000,
 * });
 *
 * connection.on(RealtimeEvents.SESSION_STARTED, (data) => {
 *     console.log("Session started");
 * });
 *
 * connection.on(RealtimeEvents.PARTIAL_TRANSCRIPT, (data) => {
 *     console.log("Partial:", data.transcript);
 * });
 *
 * connection.on(RealtimeEvents.COMMITTED_TRANSCRIPT, (data) => {
 *     console.log("Final:", data.transcript);
 *     connection.close();
 * });
 *
 * // Send audio data
 * connection.send({ audioBase64: base64String });
 *
 * // Commit and close
 * connection.commit();
 * ```
 */
export declare class RealtimeConnection {
    private websocket;
    private eventEmitter;
    private currentSampleRate;
    _audioCleanup?: () => void;
    constructor(sampleRate: number);
    /**
     * @internal
     * Used internally by ScribeRealtime to attach the WebSocket after connection is created.
     */
    setWebSocket(websocket: WebSocket): void;
    /**
     * Attaches an event listener for the specified event.
     *
     * @param event - The event to listen for (use RealtimeEvents enum)
     * @param listener - The callback function to execute when the event fires
     *
     * @example
     * ```typescript
     * connection.on(RealtimeEvents.SESSION_STARTED, (data) => {
     *     console.log("Session started", data);
     * });
     *
     * connection.on(RealtimeEvents.PARTIAL_TRANSCRIPT, (data) => {
     *     console.log("Partial:", data.transcript);
     * });
     *
     * connection.on(RealtimeEvents.COMMITTED_TRANSCRIPT, (data) => {
     *     console.log("Final:", data.transcript);
     * });
     * ```
     */
    on(event: RealtimeEvents, listener: (...args: unknown[]) => void): void;
    /**
     * Removes an event listener for the specified event.
     *
     * @param event - The event to stop listening for
     * @param listener - The callback function to remove
     *
     * @example
     * ```typescript
     * const handler = (data) => console.log(data);
     * connection.on(RealtimeEvents.PARTIAL_TRANSCRIPT, handler);
     *
     * // Later, remove the listener
     * connection.off(RealtimeEvents.PARTIAL_TRANSCRIPT, handler);
     * ```
     */
    off(event: RealtimeEvents, listener: (...args: unknown[]) => void): void;
    /**
     * Sends audio data to the transcription service.
     *
     * @param data - Audio data configuration
     * @param data.audioBase64 - Base64-encoded audio data
     * @param data.commit - Whether to commit the transcription after this chunk. You likely want to use connection.commit() instead (default: false)
     * @param data.sampleRate - Sample rate of the audio (default: configured sample rate)
     *
     * @throws {Error} If the WebSocket connection is not open
     *
     * @example
     * ```typescript
     * // Send audio chunk without committing
     * connection.send({
     *     audioBase64: base64EncodedAudio,
     * });
     *
     * // Send audio chunk with custom sample rate
     * connection.send({
     *     audioBase64: base64EncodedAudio,
     *     sampleRate: 16000,
     * });
     * ```
     */
    send(data: {
        audioBase64: string;
        commit?: boolean;
        sampleRate?: number;
    }): void;
    /**
     * Commits the transcription, signaling that a segment of audio has been sent. This clears the buffer and triggers a COMMITTED_TRANSCRIPT event. Context from previous segments is kept.
     * Committing a segment triggers a COMMITTED_TRANSCRIPT event.
     *
     * @throws {Error} If the WebSocket connection is not open
     *
     * @remarks
     * Only needed when using CommitStrategy.MANUAL.
     * When using CommitStrategy.VAD, commits are handled automatically by the server.
     *
     * @example
     * ```typescript
     * // Send all audio chunks
     * for (const chunk of audioChunks) {
     *     connection.send({ audioBase64: chunk });
     * }
     *
     * // Finalize the transcription
     * connection.commit();
     * ```
     */
    commit(): void;
    /**
     * Closes the WebSocket connection and cleans up resources.
     * This will terminate any ongoing transcription and stop microphone streaming if active.
     *
     * @remarks
     * After calling close(), this connection cannot be reused.
     * Create a new connection if you need to start transcribing again.
     *
     * @example
     * ```typescript
     * connection.on(RealtimeEvents.COMMITTED_TRANSCRIPT, (data) => {
     *     console.log("Segment committed:", data.transcript);
     *     connection.close();
     * });
     * ```
     */
    close(): void;
}
